import argparse, json, os, numpy as np, pandas as pd
from .config import load_study_config
from .parse_sparc import read_mrt_table, split_mass_models_by_galaxy, extract_columns, read_sizes_from_table1
from .windowing import select_flat_window, mesh_refine
from .ar_model import vflat_pred_ar, vflat_pred_j

def main():
    ap = argparse.ArgumentParser(description="Run T1 rotation-plateau prediction")
    ap.add_argument("--mass-models", required=True)
    ap.add_argument("--sample-table", default="")
    ap.add_argument("--sizes-csv", default="")
    ap.add_argument("--config", default="config/study.yaml")
    args = ap.parse_args()

    cfg = load_study_config(args.config)
    mm = read_mrt_table(args.mass_models)
    groups = split_mass_models_by_galaxy(mm)

    size_map={}
    if cfg.size_rule in ["RG=2.2*Rd","RG=Reff"] and args.sample_table:
        size_map = read_sizes_from_table1(args.sample_table, cfg.size_rule)
    elif cfg.size_rule=="RG=from_csv":
        import pandas as pd, os
        path = args.sizes_csv or cfg.sizes_csv
        if os.path.exists(path):
            df = pd.read_csv(path); size_map = dict(zip(df['gal_id'].astype(str), df['R_G_kpc'].astype(float)))

    rows=[]; windows={}
    for gal_id, gdf in groups.items():
        try:
            R, Vobs, comps = extract_columns(gdf)
        except Exception as e:
            continue
        i0,i1,ok = select_flat_window(R.values, Vobs.values, cfg.dvdr_max_kms_per_kpc, cfg.min_window_points, cfg.min_radius_kpc, cfg.min_window_span_kpc)
        if not ok:
            v_meas = np.nan; rmin=rmax=np.nan; mesh_pass=False
        else:
            v_meas = float(np.median(Vobs.values[i0:i1+1])); rmin=float(R.values[i0]); rmax=float(R.values[i1])
            V_ref = mesh_refine(Vobs.values, cfg.mesh_refine_factor)
            R_mid=(R.values[:-1]+R.values[1:])/2.0
            import numpy as np
            R_ref=np.empty((len(R.values)+len(R_mid),), float)
            R_ref[0::2]=R.values; R_ref[1::2]=R_mid
            dV_ref=np.gradient(V_ref, R_ref)
            mask=(abs(dV_ref)<=cfg.dvdr_max_kms_per_kpc)&(R_ref>=cfg.min_radius_kpc)
            # simplified stability: accept if any refined window exists
            mesh_pass=mask.any()

        RG = size_map.get(gal_id, np.nan)
        if RG==RG:  # finite
            if cfg.mode.upper()=="AR":
                v_pred = vflat_pred_ar(cfg.c_ms, cfg.alphaP, cfg.UGM_phys_m, cfg.R_obs_m, RG, cfg.lambda_G_default)/1000.0
            else:
                v_pred = vflat_pred_j(cfg.c_ms, cfg.J_m2, RG)/1000.0
        else:
            v_pred = np.nan

        if (v_meas==v_meas) and (v_pred==v_pred):
            abs_err = abs(v_meas - v_pred); rel_err = abs_err / max(1e-6, v_meas); claim = bool(mesh_pass)
        else:
            abs_err = np.nan; rel_err = np.nan; claim=False

        rows.append({
            "gal_id": gal_id, "RG_method": cfg.size_rule, "RG_value_m": float(RG*3.085677581491367e19) if RG==RG else np.nan,
            "chi_G": ((RG*3.085677581491367e19)**2/(cfg.UGM_phys_m*cfg.R_obs_m)) if (RG==RG and cfg.mode.upper()=="AR") else np.nan,
            "lambda_G": cfg.lambda_G_default if RG==RG else np.nan, "alphaP": cfg.alphaP if cfg.mode.upper()=="AR" else np.nan,
            "mode": cfg.mode, "v_flat_meas_kms": v_meas, "v_flat_AR_kms": v_pred, "abs_err_kms": abs_err, "rel_err": rel_err,
            "window_rmin_kpc": rmin, "window_rmax_kpc": rmax, "mesh_pass": mesh_pass, "claimable": claim
        })
        windows[gal_id]={"rmin_kpc": rmin if rmin==rmin else None, "rmax_kpc": rmax if rmax==rmax else None}
    import os, json
    os.makedirs("outputs", exist_ok=True)
    import pandas as pd
    pd.DataFrame(rows).to_csv("outputs/flat_rc_summary.csv", index=False)
    with open("outputs/windows.json","w") as f: json.dump(windows, f, indent=2)

if __name__ == "__main__":
    main()
